/*
 * Amlogic Apollo
 * tv display control driver
 *
 * Copyright (C) 2009 Amlogic, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the named License,
 * or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
 *
 * Author:   jianfeng_wang@amlogic
 *		   
 *		   
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/io.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/mutex.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/ctype.h>
#include <linux/amlogic/vout/vinfo.h>
#include <mach/am_regs.h>
#ifdef CONFIG_HIBERNATION
#include <linux/syscore_ops.h>
#endif

#if MESON_CPU_TYPE >= MESON_CPU_TYPE_MESON8
#include <mach/vpu.h>
#endif
#include <asm/uaccess.h>
#include <linux/major.h>
#include "tvconf.h"
#include "tvmode.h"
#include "vout_log.h"
#include <linux/amlogic/amlog.h>
#include <mach/power_gate.h>

#define PIN_MUX_REG_0	  0x202c
#define P_PIN_MUX_REG_0  CBUS_REG_ADDR(PIN_MUX_REG_0)
static    disp_module_info_t	disp_module_info __nosavedata;
static    disp_module_info_t    *info __nosavedata;
static void  parse_vdac_setting(char *para);

SET_TV_CLASS_ATTR(vdac_setting,parse_vdac_setting)

#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION

#define DEFAULT_POLICY_FR_AUTO	1

static int fr_auto_policy = DEFAULT_POLICY_FR_AUTO;
static int fr_auto_policy_hold = DEFAULT_POLICY_FR_AUTO;
int fps_playing_flag=0 ;//1:  23.976/29.97/59.94 fps stream is playing
vmode_t fps_target_mode=VMODE_INIT_NULL;
static void policy_framerate_automation_switch_store(char* para);
static void policy_framerate_automation_store(char* para);
SET_TV_CLASS_ATTR(policy_fr_auto, policy_framerate_automation_store)
SET_TV_CLASS_ATTR(policy_fr_auto_switch, policy_framerate_automation_switch_store)
#endif



/*****************************
*	default settings :
*	Y    -----  DAC1
*	PB  -----  DAC2
*	PR  -----  DAC0
*
*	CVBS  	---- DAC1
*	S-LUMA    ---- DAC2
*	S-CHRO	----  DAC0
******************************/

struct vmode_tvmode_tab_t {
    tvmode_t tvmode;
    vmode_t  mode;
};

static struct vmode_tvmode_tab_t mode_tab[] = {
    {TVMODE_480I, VMODE_480I},
    {TVMODE_480I_RPT, VMODE_480I_RPT},
    {TVMODE_480CVBS, VMODE_480CVBS},
    {TVMODE_480P, VMODE_480P},
    {TVMODE_480P_RPT, VMODE_480P_RPT},
    {TVMODE_576I, VMODE_576I},
    {TVMODE_576I_RPT, VMODE_576I_RPT},
    {TVMODE_576CVBS, VMODE_576CVBS},
    {TVMODE_576P, VMODE_576P},
    {TVMODE_576P_RPT, VMODE_576P_RPT},
    {TVMODE_720P, VMODE_720P},
    {TVMODE_1080I, VMODE_1080I},
    {TVMODE_1080P, VMODE_1080P},
    {TVMODE_720P_50HZ, VMODE_720P_50HZ},
    {TVMODE_1080I_50HZ, VMODE_1080I_50HZ},
    {TVMODE_1080P_50HZ, VMODE_1080P_50HZ},
    {TVMODE_1080P_24HZ, VMODE_1080P_24HZ},
    {TVMODE_4K2K_30HZ, VMODE_4K2K_30HZ},
    {TVMODE_4K2K_25HZ, VMODE_4K2K_25HZ},
    {TVMODE_4K2K_24HZ, VMODE_4K2K_24HZ},
    {TVMODE_4K2K_SMPTE, VMODE_4K2K_SMPTE},
    {TVMODE_4K2K_60HZ_Y420, VMODE_4K2K_60HZ_Y420},
    {TVMODE_4K2K_50HZ_Y420, VMODE_4K2K_50HZ_Y420},
    {TVMODE_4K2K_50HZ, VMODE_4K2K_50HZ},
    {TVMODE_VGA, VMODE_VGA},
    {TVMODE_SVGA, VMODE_SVGA},
    {TVMODE_XGA, VMODE_XGA},
    {TVMODE_SXGA, VMODE_SXGA},
    {TVMODE_WSXGA, VMODE_WSXGA},
    {TVMODE_FHDVGA, VMODE_FHDVGA},
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    {TVMODE_480P_59HZ, VMODE_480P_59HZ},
    {TVMODE_720P_59HZ, VMODE_720P_59HZ}, // for 720p 59.94hz
    {TVMODE_1080I_59HZ, VMODE_1080I_59HZ},
    {TVMODE_1080P_59HZ, VMODE_1080P_59HZ}, // for 1080p 59.94hz
    {TVMODE_1080P_23HZ, VMODE_1080P_23HZ}, // for 1080p 23.97hz
    {TVMODE_4K2K_29HZ, VMODE_4K2K_29HZ}, // for 4k2k 29.97hz
    {TVMODE_4K2K_23HZ, VMODE_4K2K_23HZ}, // for 4k2k 23.97hz
#endif
    {TVMODE_4K1K_100HZ, VMODE_4K1K_100HZ},
    {TVMODE_4K1K_100HZ_Y420, VMODE_4K1K_100HZ_Y420},
    {TVMODE_4K1K_120HZ, VMODE_4K1K_120HZ},
    {TVMODE_4K1K_120HZ_Y420, VMODE_4K1K_120HZ_Y420},
    {TVMODE_4K05K_200HZ, VMODE_4K05K_200HZ},
    {TVMODE_4K05K_200HZ_Y420, VMODE_4K05K_200HZ_Y420},
    {TVMODE_4K05K_240HZ, VMODE_4K05K_240HZ},
    {TVMODE_4K05K_240HZ_Y420, VMODE_4K05K_240HZ_Y420},
};

static const tvmode_t vmode_tvmode_map(vmode_t mode)
{
    int i = 0;

    for (i = 0; i < ARRAY_SIZE(mode_tab); i++) {
        if (mode == mode_tab[i].mode)
            return mode_tab[i].tvmode;
    }
    return TVMODE_MAX;
}

static const vinfo_t tv_info[] =
{
    { /* VMODE_480I */
        .name              = "480i",
        .mode              = VMODE_480I,
        .width             = 720,
        .height            = 480,
        .field_height      = 240,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_480I_RPT */
        .name              = "480i_rpt",
        .mode              = VMODE_480I_RPT,
        .width             = 720,
        .height            = 480,
        .field_height      = 240,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_480CVBS*/
        .name              = "480cvbs",
        .mode              = VMODE_480CVBS,
        .width             = 720,
        .height            = 480,
        .field_height      = 240,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_480P */
        .name              = "480p",
        .mode              = VMODE_480P,
        .width             = 720,
        .height            = 480,
        .field_height      = 480,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    { /* VMODE_480P_59HZ */
        .name              = "480p59hz",
        .mode              = VMODE_480P_59HZ,
        .width             = 720,
        .height            = 480,
        .field_height      = 480,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60000,
        .sync_duration_den = 1001,
        .video_clk         = 27000000,
    },
#endif
    { /* VMODE_480P_RPT */
        .name              = "480p_rpt",
        .mode              = VMODE_480P_RPT,
        .width             = 720,
        .height            = 480,
        .field_height      = 480,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_576I */
        .name              = "576i",
        .mode              = VMODE_576I,
        .width             = 720,
        .height            = 576,
        .field_height      = 288,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_576I_RPT */
        .name              = "576i_rpt",
        .mode              = VMODE_576I_RPT,
        .width             = 720,
        .height            = 576,
        .field_height      = 288,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_576I */
        .name              = "576cvbs",
        .mode              = VMODE_576CVBS,
        .width             = 720,
        .height            = 576,
        .field_height      = 288,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_576P */
        .name              = "576p",
        .mode              = VMODE_576P,
        .width             = 720,
        .height            = 576,
        .field_height      = 576,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_576P_RPT */
        .name              = "576p_rpt",
        .mode              = VMODE_576P_RPT,
        .width             = 720,
        .height            = 576,
        .field_height      = 576,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 27000000,
    },
    { /* VMODE_720P */
        .name              = "720p",
        .mode              = VMODE_720P,
        .width             = 1280,
        .height            = 720,
        .field_height      = 720,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 74250000,
    },
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    { /* VMODE_720P_59HZ */
        .name              = "720p59hz",
        .mode              = VMODE_720P_59HZ,
        .width             = 1280,
        .height            = 720,
        .field_height      = 720,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60000,
        .sync_duration_den = 1001,
        .video_clk         = 74250000,
    },
#endif
    { /* VMODE_1080I */
        .name              = "1080i",
        .mode              = VMODE_1080I,
        .width             = 1920,
        .height            = 1080,
        .field_height      = 540,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 74250000,
    },
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    { /* VMODE_1080I_59HZ */
        .name              = "1080i59hz",
        .mode              = VMODE_1080I_59HZ,
        .width             = 1920,
        .height            = 1080,
        .field_height      = 540,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60000,
        .sync_duration_den = 1001,
        .video_clk         = 74250000,
    },
#endif
    { /* VMODE_1080P */
        .name              = "1080p",
        .mode              = VMODE_1080P,
        .width             = 1920,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 148500000,
    },
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    { /* VMODE_1080P_59HZ */
        .name			   = "1080p59hz",
        .mode			   = VMODE_1080P_59HZ,
        .width			   = 1920,
        .height 		   = 1080,
        .field_height	   = 1080,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60000,
        .sync_duration_den = 1001,
        .video_clk		   = 148500000,
    },
#endif
    { /* VMODE_720P_50hz */
        .name              = "720p50hz",
        .mode              = VMODE_720P_50HZ,
        .width             = 1280,
        .height            = 720,
        .field_height      = 720,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 74250000,
    },
    { /* VMODE_1080I_50HZ */
        .name              = "1080i50hz",
        .mode              = VMODE_1080I_50HZ,
        .width             = 1920,
        .height            = 1080,
        .field_height      = 540,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 74250000,
    },
    { /* VMODE_1080P_50HZ */
        .name              = "1080p50hz",
        .mode              = VMODE_1080P_50HZ,
        .width             = 1920,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 148500000,
    },
    { /* VMODE_1080P_24HZ */
        .name              = "1080p24hz",
        .mode              = VMODE_1080P_24HZ,
        .width             = 1920,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 24,
        .sync_duration_den = 1,
        .video_clk         = 74250000,
    },
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    { /* VMODE_1080P_23HZ */
        .name			   = "1080p23hz",
        .mode			   = VMODE_1080P_23HZ,
        .width			   = 1920,
        .height 		   = 1080,
        .field_height	   = 1080,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 24000,
        .sync_duration_den = 1001,
        .video_clk		   = 74250000,
	},
#endif
    { /* VMODE_4K2K_30HZ */
        .name              = "4k2k30hz",
        .mode              = VMODE_4K2K_30HZ,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 30,
        .sync_duration_den = 1,
        .video_clk         = 297000000,
    },
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    { /* VMODE_4K2K_29HZ */
        .name			   = "4k2k29hz",
        .mode			   = VMODE_4K2K_29HZ,
        .width			   = 3840,
        .height 		   = 2160,
        .field_height	   = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 2997,
        .sync_duration_den = 100,
        .video_clk		   = 297000000,
	},
#endif
    { /* VMODE_4K2K_25HZ */
        .name              = "4k2k25hz",
        .mode              = VMODE_4K2K_25HZ,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 25,
        .sync_duration_den = 1,
        .video_clk         = 297000000,
    },
    { /* VMODE_4K2K_24HZ */
        .name              = "4k2k24hz",
        .mode              = VMODE_4K2K_24HZ,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 24,
        .sync_duration_den = 1,
        .video_clk         = 297000000,
    },
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
    { /* VMODE_4K2K_23HZ */
        .name			   = "4k2k23hz",
        .mode			   = VMODE_4K2K_23HZ,
        .width			   = 3840,
        .height 		   = 2160,
        .field_height	   = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 24000,
        .sync_duration_den = 1001,
        .video_clk         = 297000000,
    },
#endif
    { /* VMODE_4K2K_SMPTE */
        .name              = "4k2ksmpte",
        .mode              = VMODE_4K2K_SMPTE,
        .width             = 4096,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 24,
        .sync_duration_den = 1,
        .video_clk         = 297000000,
    },
    { /* VMODE_4K2K_FAKE_5G */
        .name              = "4k2k5g",
        .mode              = VMODE_4K2K_FAKE_5G,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 495000000,
    },
    { /* VMODE_4K2K_60HZ_Y420 */
        .name              = "4k2k60hz420",
        .mode              = VMODE_4K2K_60HZ_Y420,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K2K_50HZ_Y420 */
        .name              = "4k2k50hz420",
        .mode              = VMODE_4K2K_50HZ_Y420,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K2K_60HZ */
        .name              = "4k2k60hz",
        .mode              = VMODE_4K2K_60HZ,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K1K_100HZ_Y420 */
        .name              = "4k1k100hz420",
        .mode              = VMODE_4K1K_100HZ_Y420,
        .width             = 3840,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 32,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 100,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K1K_100HZ */
        .name              = "4k1k100hz",
        .mode              = VMODE_4K1K_100HZ,
        .width             = 3840,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 32,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 100,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K1K_120HZ_Y420 */
        .name              = "4k1k120hz420",
        .mode              = VMODE_4K1K_120HZ_Y420,
        .width             = 3840,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 32,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 120,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K1K_120HZ */
        .name              = "4k1k120hz",
        .mode              = VMODE_4K1K_120HZ,
        .width             = 3840,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 32,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 120,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K05K_200HZ_Y420 */
        .name              = "4k05k200hz420",
        .mode              = VMODE_4K05K_200HZ_Y420,
        .width             = 3840,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 64,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 200,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K05K_200HZ */
        .name              = "4k05k200hz",
        .mode              = VMODE_4K05K_200HZ,
        .width             = 3840,
        .height            = 540,
        .field_height      = 540,
        .aspect_ratio_num  = 64,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 200,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K05K_240HZ_Y420 */
        .name              = "4k05k240hz420",
        .mode              = VMODE_4K05K_240HZ_Y420,
        .width             = 3840,
        .height            = 540,
        .field_height      = 540,
        .aspect_ratio_num  = 64,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 240,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K05K_240HZ */
        .name              = "4k05k240hz",
        .mode              = VMODE_4K05K_240HZ,
        .width             = 3840,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 64,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 240,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_4K2K_50HZ */
        .name              = "4k2k50hz",
        .mode              = TVMODE_4K2K_50HZ,
        .width             = 3840,
        .height            = 2160,
        .field_height      = 2160,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 50,
        .sync_duration_den = 1,
        .video_clk         = 594000000,
    },
    { /* VMODE_vga */
        .name              = "vga",
        .mode              = VMODE_VGA,
        .width             = 640,
        .height            = 480,
        .field_height      = 240,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 25175000,
    },
    { /* VMODE_SVGA */
        .name              = "svga",
        .mode              = VMODE_SVGA,
        .width             = 800,
        .height            = 600,
        .field_height      = 600,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 40000000,
    },
    { /* VMODE_XGA */
        .name              = "xga",
        .mode              = VMODE_XGA,
        .width             = 1024,
        .height            = 768,
        .field_height      = 768,
        .aspect_ratio_num  = 4,
        .aspect_ratio_den  = 3,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 65000000,
    },
    { /* VMODE_sxga */
        .name              = "sxga",
        .mode              = VMODE_SXGA,
        .width             = 1280,
        .height            = 1024,
        .field_height      = 1024,
        .aspect_ratio_num  = 5,
        .aspect_ratio_den  = 4,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 108000000,
    },
    { /* VMODE_wsxga */
        .name              = "wsxga",
        .mode              = VMODE_WSXGA,
        .width             = 1440,
        .height            = 900,
        .field_height      = 900,
        .aspect_ratio_num  = 8,
        .aspect_ratio_den  = 5,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 88750000,
    },
    { /* VMODE_fhdvga */
        .name              = "fhdvga",
        .mode              = VMODE_FHDVGA,
        .width             = 1920,
        .height            = 1080,
        .field_height      = 1080,
        .aspect_ratio_num  = 16,
        .aspect_ratio_den  = 9,
        .sync_duration_num = 60,
        .sync_duration_den = 1,
        .video_clk         = 148500000,
    },
};

static const struct file_operations am_tv_fops = {
    .open	= NULL,
    .read	= NULL,//am_tv_read,
    .write	= NULL,
    .unlocked_ioctl	= NULL,//am_tv_ioctl,
    .release	= NULL,
    .poll		= NULL,
};

static const vinfo_t *get_valid_vinfo(char  *mode)
{
    const vinfo_t * vinfo = NULL;
    int  i,count=ARRAY_SIZE(tv_info);
    int mode_name_len=0;

    for (i=0;i<count;i++)
    {
        if (strncmp(tv_info[i].name,mode,strlen(tv_info[i].name)) == 0)
        {
            if ((vinfo == NULL) || (strlen(tv_info[i].name)>mode_name_len)) {
                vinfo = &tv_info[i];
                mode_name_len = strlen(tv_info[i].name);
            }
        }
    }
    return vinfo;
}

static const vinfo_t *tv_get_current_info(void)
{
    return info->vinfo;
}

tvmode_t vmode_to_tvmode(vmode_t mod)
{
    return vmode_tvmode_map(mod);
}

static const vinfo_t *get_tv_info(vmode_t mode)
{
    int i = 0;
    for (i = 0; i < ARRAY_SIZE(tv_info); i++) {
        if (mode == tv_info[i].mode)
            return &tv_info[i];
    }
    return NULL;
}

static int tv_set_current_vmode(vmode_t mod)
{
    if ((mod&VMODE_MODE_BIT_MASK)> VMODE_MAX)
        return -EINVAL;
    info->vinfo = get_tv_info(mod & VMODE_MODE_BIT_MASK);
    if (!info->vinfo) {
        printk("don't get tv_info, mode is %d\n", mod);
        return 1;
    }
//	info->vinfo = &tv_info[mod & VMODE_MODE_BIT_MASK];
    printk("mode is %d,sync_duration_den=%d,sync_duration_num=%d\n", mod,info->vinfo->sync_duration_den,info->vinfo->sync_duration_num);
    if (mod&VMODE_LOGO_BIT_MASK)  return 0;
#if MESON_CPU_TYPE >= MESON_CPU_TYPE_MESON8
    switch_vpu_mem_pd_vmod(info->vinfo->mode, VPU_MEM_POWER_ON);
    request_vpu_clk_vmod(info->vinfo->video_clk, info->vinfo->mode);
#endif
    tvoutc_setmode(vmode_to_tvmode(mod));
//	change_vdac_setting(get_current_vdac_setting(),mod);
    return 0;
}

static vmode_t tv_validate_vmode(char *mode)
{
    const vinfo_t *info = get_valid_vinfo(mode);
    if (info)
        return info->mode;

    return VMODE_MAX;
}
static int tv_vmode_is_supported(vmode_t mode)
{
    int  i,count=ARRAY_SIZE(tv_info);
    mode&=VMODE_MODE_BIT_MASK;
    for (i=0;i<count;i++)
    {
        if (tv_info[i].mode == mode)
        {
            return true;
        }
    }
    return false;
}
static int tv_module_disable(vmode_t cur_vmod)
{
#if MESON_CPU_TYPE >= MESON_CPU_TYPE_MESON8
    if (info->vinfo) {
        release_vpu_clk_vmod(info->vinfo->mode);
        switch_vpu_mem_pd_vmod(info->vinfo->mode, VPU_MEM_POWER_DOWN);
    }
#endif
    //video_dac_disable();
    return 0;
}

#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
char* get_name_from_vmode(vmode_t mode)
{
    int i = 0, count = 0;

    count = ARRAY_SIZE(tv_info);
    for ( i=0; i<count; i++ )
    {
        if ( tv_info[i].mode == mode )
            break;
    }

    if ( i == count )
        return NULL;

    return tv_info[i].name;
}

// frame_rate = 9600/duration/100 hz
static int get_vsource_frame_rate(int duration)
{
    int frame_rate = 0;

    switch ( duration )
    {
        case 1600:
            frame_rate = 6000;
            break;
        case 1601:
        case 1602:
            frame_rate = 5994;
            break;
        case 1920:
            frame_rate = 5000;
            break;
        case 3200:
            frame_rate = 3000;
            break;
        case 3203:
            frame_rate = 2997;
            break;
        case 3840:
            frame_rate = 2500;
            break;
        case 4000:
            frame_rate = 2400;
            break;
        case 4004:
            frame_rate = 2397;
            break;
        default:
            break;
    }

    return frame_rate;
}

static int get_target_frame_rate(int framerate_vsource, int policy)
{
    const vinfo_t *pvinfo ;
    int framerate_target = 0;
    pvinfo = tv_get_current_info();
    switch ( policy )
    {
        case 0: // not change
            break;
        case 1: // change to the frame rate of video source
            if ( (framerate_vsource == 2397) || (framerate_vsource == 2997) || (framerate_vsource == 5994) ) {
                switch (pvinfo->sync_duration_num)
                {
                case 24:
                    framerate_target=2397;
                    break;
                case 30:
                    framerate_target=2997;
                    break;
                case 60:
                    framerate_target=5994;
                    break;
                default:
                    framerate_target = ( pvinfo->sync_duration_num > 100 ? pvinfo->sync_duration_num : pvinfo->sync_duration_num*100 );
                    break;
                }
            }
            else
                framerate_target = framerate_vsource;
            break;
        case 2: // change to the frame rate of video source, but use 59.94 for 23.97/29.97
            if ( (framerate_vsource == 2397) || (framerate_vsource == 2997) )
                framerate_target = 5994;
            else
                framerate_target = framerate_vsource;
            break;
        default:
            break;
    }
    return framerate_target;
}

//extern int hdmitx_is_vmode_supported(char *mode_name);

static int get_target_vmode(int framerate_target)
{
//	int is_receiver_supported = 0;
    const vinfo_t *pvinfo ;
    vmode_t mode_target = VMODE_INIT_NULL;

    printk("vout [%s] frame_rate_target = %d\n", __FUNCTION__, framerate_target);

    pvinfo = tv_get_current_info();

    mode_target = pvinfo->mode;

    if ( (framerate_target == 2397) || (framerate_target == 2997) || (framerate_target == 5994) ) {
        switch ( mode_target ) {
            case VMODE_480P:
                mode_target = VMODE_480P_59HZ;
                break;
            case VMODE_720P:
                mode_target = VMODE_720P_59HZ;
                break;
            case VMODE_1080I:
                mode_target = VMODE_1080I_59HZ;
                break;
            case VMODE_1080P_24HZ:
                mode_target = VMODE_1080P_23HZ;
                break;
            case VMODE_1080P:
                mode_target = VMODE_1080P_59HZ;
                break;
            case VMODE_4K2K_24HZ:
                mode_target = VMODE_4K2K_23HZ;
                break;
            case VMODE_4K2K_30HZ:
                mode_target = VMODE_4K2K_29HZ;
                break;
            default:
                break;
        }
    }
/*
    is_receiver_supported = hdmitx_is_vmode_supported(get_name_from_vmode(mode_target));

    switch ( is_receiver_supported )
    {
        case 0: // not supported in edid
            mode_target = pvinfo->mode;
            break;
        case 1: // supported in edid
            break;
        case 2: // no edid
            mode_target = pvinfo->mode;
            break;
        default:
            break;
    }
*/
    fps_target_mode=mode_target;
    return mode_target;
}

// return values:
//		0: 		same vmode, need not change
//		1: 		similar vmode, just change pll to add 0.1% clock
//		0xff: 	similar vmode, just change pll to reduce 0.1% clock
//		2: 		different vmode, need change mode
static int get_exchange_mode(vmode_t mode_target)
{
    const vinfo_t *pvinfo;
    vmode_t mode_current = VMODE_INIT_NULL;

    pvinfo = tv_get_current_info();
    mode_current = pvinfo->mode;

    if ( mode_current == mode_target )
        return 0;

    if ( ((mode_current == VMODE_480P) && (mode_target == VMODE_480P_59HZ)) ||
        ((mode_current == VMODE_480P_59HZ) && (mode_target == VMODE_480P)) ||
        ((mode_current == VMODE_720P) && (mode_target == VMODE_720P_59HZ)) ||
        ((mode_current == VMODE_720P_59HZ) && (mode_target == VMODE_720P)) ||
        ((mode_current == VMODE_1080I) && (mode_target == VMODE_1080I_59HZ)) ||
        ((mode_current == VMODE_1080I_59HZ) && (mode_target == VMODE_1080I)) ||
        ((mode_current == VMODE_1080P) && (mode_target == VMODE_1080P_59HZ)) ||
        ((mode_current == VMODE_1080P_59HZ) && (mode_target == VMODE_1080P)) ||
        ((mode_current == VMODE_1080P_24HZ) && (mode_target == VMODE_1080P_23HZ)) ||
        ((mode_current == VMODE_1080P_23HZ) && (mode_target == VMODE_1080P_24HZ) ) ||
        ((mode_current == VMODE_4K2K_30HZ) && (mode_target == VMODE_4K2K_29HZ)) ||
        ((mode_current == VMODE_4K2K_29HZ) && (mode_target == VMODE_4K2K_30HZ)) ||
        ((mode_current == VMODE_4K2K_24HZ) && (mode_target == VMODE_4K2K_23HZ)) ||
        ((mode_current == VMODE_4K2K_23HZ) && (mode_target == VMODE_4K2K_24HZ)) )
        return 0x1;

    return 2;
}

extern int hdmitx_is_special_tv_process(void);

// just to fine tune the 0.1% clock
static int clock_fine_tune(void)
{
    const vinfo_t *pvinfo ;
    pvinfo = tv_get_current_info();
#if (MESON_CPU_TYPE == MESON_CPU_TYPE_MESON8) || (MESON_CPU_TYPE == MESON_CPU_TYPE_MESON8B)
    switch ( pvinfo->mode )
    {
        case VMODE_720P_59HZ:
        case VMODE_1080I_59HZ:
        case VMODE_1080P_23HZ:
        case VMODE_1080P_59HZ:
            if ( MESON_CPU_TYPE == MESON_CPU_TYPE_MESON8B )
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c84cf8);
            else
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c84d04);
            break;
        case VMODE_4K2K_23HZ:
        case VMODE_4K2K_29HZ:
            if ( MESON_CPU_TYPE == MESON_CPU_TYPE_MESON8B )
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69d84cf8);
            else if (IS_MESON_M8_CPU && hdmitx_is_special_tv_process())//SAMSUNG future TV, M8, in 4K2K mode
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x59c84d04);
            else
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69d84d04);
            break;
        case VMODE_720P:
        case VMODE_1080I:
        case VMODE_1080P:
        case VMODE_1080P_24HZ:
            aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c84e00);
            break;
        case VMODE_4K2K_24HZ:
        case VMODE_4K2K_30HZ:
            if (IS_MESON_M8_CPU && hdmitx_is_special_tv_process())//SAMSUNG future TV, M8, in 4K2K mode
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x59c84e00);
            else
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c84e00);
            break;
        case VMODE_480P_59HZ:
            if ( (MESON_CPU_TYPE == MESON_CPU_TYPE_MESON8B) || (IS_MESON_M8M2_CPU) ) {
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c84f48);
                aml_write_reg32(P_HHI_VID_PLL_CNTL,  0x400d042c);
            }
            else {
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c8cf48);
                aml_write_reg32(P_HHI_VID_PLL_CNTL,  0x4008042c);
            }
            break;
        case VMODE_480P:
            if ( (MESON_CPU_TYPE == MESON_CPU_TYPE_MESON8B) || (IS_MESON_M8M2_CPU) ) {
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c84000);
                aml_write_reg32(P_HHI_VID_PLL_CNTL,  0x400d042d);
            }
            else {
                aml_write_reg32(P_HHI_VID_PLL_CNTL2, 0x69c88000);
                aml_write_reg32(P_HHI_VID_PLL_CNTL,  0x4008042d);
            }
            break;
        default:
            break;
    }

#endif

    return 0;
}

extern void update_vmode_status(char* name);
extern void set_vout_mode_fr_auto(char * name);

static void update_current_vinfo(vmode_t mode)
{
	if ((mode&VMODE_MODE_BIT_MASK)> VMODE_FHDVGA)
		return ;

	info->vinfo = &tv_info[mode & VMODE_MODE_BIT_MASK];

	return ;
}

static int framerate_automation_set_mode(vmode_t mode_target)
{
	int auto_mode = 0;

	auto_mode = get_exchange_mode(mode_target);

	printk("vout [%s] mode_target = %d\n", __FUNCTION__, mode_target);
	printk("+++++++++++++++++%s[%d]auto_mode=%d++++++++++++++++\n",__func__,__LINE__,auto_mode);
	switch( auto_mode )
	{
		case 0:
			// need not change vmode
			break;
		case 1:
			// just change pll to adjust clock
		 	update_vmode_status(get_name_from_vmode(mode_target));
			update_current_vinfo(mode_target);
			clock_fine_tune();
			vout_notifier_call_chain(VOUT_EVENT_MODE_CHANGE,&mode_target) ;
			break;
		case 2:
			// change vmode and notify all client
			set_vout_mode_fr_auto(get_name_from_vmode(mode_target));
			break;
		default:
			break;
	}

	return 0;
}

static int framerate_automation_process(int duration)
{
	int policy=0, fr_vsource = 0, fr_target = 0;
	vmode_t mode_target = VMODE_INIT_NULL;
	const vinfo_t *pvinfo;

	printk("vout [%s] duration = %d\n", __FUNCTION__, duration);
	policy = fr_auto_policy;
	if( policy == 0 )
	{
		printk("vout frame rate automation disabled!\n");
		return 1;
	}

	fr_vsource = get_vsource_frame_rate(duration);
	fr_target = get_target_frame_rate(fr_vsource, policy);
	
	pvinfo = tv_get_current_info();
	if( (pvinfo->sync_duration_num==fr_target) || (pvinfo->sync_duration_num==(fr_target/100)) )
		return 0;
	switch(fr_vsource){
	case 5994:
	case 2997:
	case 2397:
    	fps_playing_flag=1;
		break;
	default:
		fps_playing_flag=0;
		break;
	}
	printk("%s[%d] fps_playing_flag = %d\n", __FUNCTION__,__LINE__, fps_playing_flag);
	mode_target = get_target_vmode(fr_target);
	
	framerate_automation_set_mode(mode_target);

	return 0;
}

#endif
static int tv_set_vframe_rate_hint(int duration)
{
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION

	printk("vout [%s] duration = %d, policy = %d!\n", __FUNCTION__, duration, fr_auto_policy);

	framerate_automation_process(duration);

#endif

	return 0;
}

#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
extern vmode_t mode_by_user ;
#endif

static int tv_set_vframe_rate_end_hint(void)
{
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION

	printk("vout [%s] return mode = %d, policy = %d!\n", __FUNCTION__, mode_by_user, fr_auto_policy);
	if( fr_auto_policy != 0 )
	{
		fps_playing_flag=0;
		framerate_automation_set_mode(mode_by_user);
	}

#endif

	return 0;
}

#ifdef  CONFIG_PM
#if MESON_CPU_TYPE >= MESON_CPU_TYPE_MESON8
extern void cvbs_cntl_output(unsigned int open);
#endif
static int tv_suspend(int pm_event)
{
	/* in freeze process do not turn off the display devices */
	if (pm_event == PM_EVENT_FREEZE)
		return 0;
	video_dac_disable();
#if MESON_CPU_TYPE >= MESON_CPU_TYPE_MESON8
	cvbs_cntl_output(0);
#endif

	return 0;
}
static int tv_resume(int pm_event)
{
	/* in thaw/restore process do not reset the display mode */
	if (pm_event == PM_EVENT_THAW
		|| pm_event == PM_EVENT_RESTORE)
		return 0;
	video_dac_enable(0xff);
	tv_set_current_vmode(info->vinfo->mode);
	return 0;
}
#endif 
static vout_server_t tv_server={
	.name = "vout_tv_server",
	.op = {	
		.get_vinfo=tv_get_current_info,
		.set_vmode=tv_set_current_vmode,
		.validate_vmode=tv_validate_vmode,
		.vmode_is_supported=tv_vmode_is_supported,
		.disable = tv_module_disable,
		.set_vframe_rate_hint = tv_set_vframe_rate_hint,
		.set_vframe_rate_end_hint = tv_set_vframe_rate_end_hint,
#ifdef  CONFIG_PM  
		.vout_suspend=tv_suspend,
		.vout_resume=tv_resume,
#endif	
	},
};

static void _init_vout(void)
{
	if (info->vinfo == NULL)
		info->vinfo = &tv_info[TVMODE_720P];
}

/***************************************************
**
**	The first digit control component Y output DAC number
**	The 2nd digit control component U output DAC number
**	The 3rd digit control component V output DAC number
**	The 4th digit control composite CVBS output DAC number
**	The 5th digit control s-video Luma output DAC number
**	The 6th digit control s-video chroma output DAC number
** 	examble :
**		echo  120120 > /sys/class/display/vdac_setting
**		the first digit from the left side .	
******************************************************/
static void  parse_vdac_setting(char *para) 
{
	int  i;
	char  *pt=strstrip(para);
	int len=strlen(pt);
	u32  vdac_sequence=get_current_vdac_setting();
	
	amlog_mask_level(LOG_MASK_PARA,LOG_LEVEL_LOW,"origin vdac setting:0x%x,strlen:%d\n",vdac_sequence,len);
	if(len!=6)
	{
		amlog_mask_level(LOG_MASK_PARA,LOG_LEVEL_HIGH,"can't parse vdac settings\n");
		return ;
	}
	vdac_sequence=0;
	for(i=0;i<6;i++)
	{
		vdac_sequence<<=4;
		vdac_sequence|=*pt -'0';
		pt++;
	}
	amlog_mask_level(LOG_MASK_PARA,LOG_LEVEL_LOW,"current vdac setting:0x%x\n",vdac_sequence);
	
	change_vdac_setting(vdac_sequence,get_current_vmode());
}

#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
//
// 0: disable frame_rate_automation feature
// 1: enable frame_rate_automation feature, same with frame rate of video source
// 2: enable frame_rate_automation feature, use 59.94 instead of 23.97/29.97
// 3: reset frame_rate_automatio
static void policy_framerate_automation_store(char* para)
{
    int policy = 0;

    policy = simple_strtoul(para, NULL, 10);

    if ((policy >=0 ) && (policy < 3))
    {
        fr_auto_policy_hold = policy;
        fr_auto_policy = fr_auto_policy_hold;
        snprintf(policy_fr_auto_switch, 40, "%d\n", fr_auto_policy);
    }
    snprintf(policy_fr_auto, 40, "%d\n", fr_auto_policy_hold);

    return ;
}

static void policy_framerate_automation_switch_store(char* para)
{
    int policy = 0;
    policy = simple_strtoul(para, NULL, 10);

    if ((policy >= 0) && (policy < 3))
    {
        fr_auto_policy = policy;
    }
    else if (policy == 3)
    {
        fr_auto_policy = fr_auto_policy_hold;
        tv_set_vframe_rate_end_hint();
    }
    snprintf(policy_fr_auto_switch, 40, "%d\n", fr_auto_policy);
   return ;
}

#endif

static  struct  class_attribute   *tv_attr[]={
&class_TV_attr_vdac_setting,
#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
&class_TV_attr_policy_fr_auto,
&class_TV_attr_policy_fr_auto_switch,
#endif
};
static int  create_tv_attr(disp_module_info_t* info)
{
	//create base class for display
	int  i;

	info->base_class=class_create(THIS_MODULE,info->name);
	if(IS_ERR(info->base_class))
	{
		amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"create tv display class fail\n");
		return  -1 ;
	}
	//create  class attr
	for(i=0;i<ARRAY_SIZE(tv_attr);i++)
	{
		if ( class_create_file(info->base_class,tv_attr[i]))
		{
			amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"create disp attribute %s fail\n",tv_attr[i]->attr.name);
		}
	}
	sprintf(vdac_setting,"%x",get_current_vdac_setting());

#ifdef CONFIG_AML_VOUT_FRAMERATE_AUTOMATION
	sprintf(policy_fr_auto, "%d", DEFAULT_POLICY_FR_AUTO);
  sprintf(policy_fr_auto_switch, "%d", DEFAULT_POLICY_FR_AUTO);
#endif

	return   0;
}

#ifdef CONFIG_HIBERNATION
struct class  *info_base_class;
static int tvconf_suspend(void)
{
	info_base_class = info->base_class;
	return 0;
}

static void tvconf_resume(void)
{
	info->base_class = info_base_class;
}

static struct syscore_ops tvconf_ops = {
	.suspend = tvconf_suspend,
	.resume = tvconf_resume,
	.shutdown = NULL,
};
#endif

static int __init tv_init_module(void)
{
	int  ret ;

#ifdef CONFIG_HIBERNATION
	INIT_LIST_HEAD(&tvconf_ops.node);
	register_syscore_ops(&tvconf_ops);
#endif

	info=&disp_module_info;
	printk("%s\n", __func__);

	/*if (!info)
	{
		amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"can't alloc display info struct\n");
		return -ENOMEM;
	}*/
	
	memset(info, 0, sizeof(disp_module_info_t));

	sprintf(info->name,TV_CLASS_NAME) ;
	ret=register_chrdev(0,info->name,&am_tv_fops);
	if(ret <0) 
	{
		amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"register char dev tv error\n");
		return  ret ;
	}
	info->major=ret;
	_init_vout();
	amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"major number %d for disp\n",ret);
	if(vout_register_server(&tv_server))
	{
		amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"register tv module server fail\n");
	}
	else
	{
		amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"register tv module server ok\n");
	}
	create_tv_attr(info);
	return 0;

}
static __exit void tv_exit_module(void)
{
	int i;
	
	if(info->base_class)
	{
		for(i=0;i<ARRAY_SIZE(tv_attr);i++)
		{
			class_remove_file(info->base_class,tv_attr[i]) ;
		}
		
		class_destroy(info->base_class);
	}	
	if(info)
	{
		unregister_chrdev(info->major,info->name)	;
		//kfree(info);
	}
	vout_unregister_server(&tv_server);
	
	amlog_mask_level(LOG_MASK_INIT,LOG_LEVEL_HIGH,"exit tv module\n");
}

#if (MESON_CPU_TYPE >= MESON_CPU_TYPE_MESON8)
extern void cvbs_config_vdac(unsigned int flag, unsigned int cfg);

static int __init vdac_config_bootargs_setup(char* line)
{
    unsigned int cfg = 0x00;

    printk("cvbs trimming line = %s\n", line);
    cfg = simple_strtoul(line, NULL, 16);

    cvbs_config_vdac((cfg&0xff00)>>8, cfg&0xff);

    return 1;
}

__setup("vdaccfg=", vdac_config_bootargs_setup);
#endif

extern void cvbs_performance_config(unsigned int index);
static int __init cvbs_performance_setup(char* line)
{
	unsigned int cfg = 0x1;

	printk("cvbs performance line = %s\n", line);
	cfg = simple_strtoul(line, NULL, 10);

	cvbs_performance_config(cfg);
	return 0;
}
__setup("cvbsdrv=", cvbs_performance_setup);

arch_initcall(tv_init_module);
module_exit(tv_exit_module);

MODULE_DESCRIPTION("display configure  module");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("jianfeng_wang <jianfeng.wang@amlogic.com>");

